/****************************************************
 * siliCAT.c
 *
 * siliCAT stack 1.1
 * for communication with SLC1044 and SLC1045
 *
 * 31.12.2008
 * G. Bartelt
 * www.sebulli.com
 ****************************************************/

#include "siliCAT.h"

/**********************  PRIVATE FUNCTIONS   *********************/

/****************************************************
 * encode one layer-3 databyte to two layer-2 bytes
 ****************************************************/
void siliCAT_TxDataByte(unsigned char b, unsigned char isfirst)
{
 if (isfirst)
  siliCAT_TxByte(siliCAT_1stDATABYTE | ((b>>6)&0x03));
 else
  siliCAT_TxByte(siliCAT_nthDATABYTE | ((b>>6)&0x03));

 siliCAT_TxByte(siliCAT_BYTELSB | (b & 0x3F));
}


/****************************************************
 * encode layer-3 siliCAT ID and data type
 ****************************************************/
void siliCAT_TxIDTyp(unsigned int id,unsigned char siliCATTyp)
{
 siliCAT_TxDataByte(id & 0x00FF,1);
 siliCAT_TxDataByte((id>>8 & 0x000F)|(siliCATTyp<<4),0);
}


/****************************************************
 * encode string data (lenght + characters)
 ****************************************************/
void siliCAT_TxString(char* s)
{
int siliCATStrLen;
int i;

  siliCATStrLen=0;
  if (s!=0)
    for (i=0;(i<255)&&(s[i]!='\0');i++)
      siliCATStrLen++;

  siliCAT_TxDataByte(siliCATStrLen & 0x00FF,0);

  for (i=0;i<siliCATStrLen;i++)
    siliCAT_TxDataByte(s[i],0);
}


/****************************************************
 * encode byte array (lenght + bytes)
 ****************************************************/
void siliCAT_TxBytes(unsigned char l,char* bytes)
{
int i;

  siliCAT_TxDataByte(l,0);

  for (i=0;i<l;i++)
    siliCAT_TxDataByte(bytes[i],0);
}


/****************************************************
 * encode list command and data
 ****************************************************/
void siliCAT_List(unsigned int id,char listcommand,char* s)
{
  siliCAT_TxIDTyp(id,siliCAT_TYPE_LIST);
  siliCAT_TxDataByte(listcommand ,0);
  siliCAT_TxString(s);
}







/**********************  PUBLIC FUNCTIONS   *********************/

/****************************************************
 * send one signed byte value
 ****************************************************/
void siliCAT_SetSignedByte(unsigned int id,signed char sb)
{
 siliCAT_TxIDTyp(id,siliCAT_TYPE_BYTE);
 siliCAT_TxDataByte(sb,0);
}


/****************************************************
 * send one signed integer value
 ****************************************************/
void siliCAT_SetSignedInt(unsigned int id,signed int si)
{
 siliCAT_TxIDTyp(id,siliCAT_TYPE_INT);
 siliCAT_TxDataByte(si & 0x00FF,0);
 siliCAT_TxDataByte((si>>8) & 0x00FF,0);
}


/****************************************************
 * send one signed long value
 ****************************************************/
void siliCAT_SetSignedLong(unsigned int id,signed long sl)
{
 siliCAT_TxIDTyp(id,siliCAT_TYPE_LONG);
 siliCAT_TxDataByte(sl & 0x000000FF,0);
 siliCAT_TxDataByte((sl>>8) & 0x000000FF,0);
 siliCAT_TxDataByte((sl>>16) & 0x000000FF,0);
 siliCAT_TxDataByte((sl>>24) & 0x000000FF,0);
}


/****************************************************
 * send one string value
 ****************************************************/
void siliCAT_SetString(unsigned int id, char* s)
{
  siliCAT_TxIDTyp(id,3);
  siliCAT_TxString(s);
}


/****************************************************
 * send signed byte values
 ****************************************************/
void siliCAT_SetSignedBytes(unsigned int id,unsigned char l,signed char* bytes)
{
 siliCAT_TxIDTyp(id,siliCAT_TYPE_BYTES);
 siliCAT_TxBytes(l,bytes);
}


/****************************************************
 * send visibility of one object
 ****************************************************/
void siliCAT_SetVisibility(unsigned int id,unsigned char v)
{
  siliCAT_TxIDTyp(id,siliCAT_TYPE_VISIBILITY);
  siliCAT_TxDataByte(v & 0x01,0);
}


/****************************************************
 * add item to list object
 ****************************************************/
void siliCAT_ListAdd(unsigned int id,char* s)
{
 siliCAT_List(id,'A',s);
}


/****************************************************
 * remove item from list object
 ****************************************************/
void siliCAT_ListRemove(unsigned int id,char* s)
{
 siliCAT_List(id,'R',s);
}


/****************************************************                                       
 * clear list
 ****************************************************/
void siliCAT_ListClear(unsigned int id)
{
 siliCAT_List(id,'C',0);
}







/****************************************************                                       
 * receive one byte and decode it
 ****************************************************/
void siliCAT_RxByte(unsigned char b)
{
static unsigned char rxstate=0;
static unsigned char db;
static unsigned int id;
static unsigned char siliCATTyp;
static unsigned long siliCATValue;
static char rxstrbuf[145];
static unsigned char rxstrbufptr;
static unsigned char rxstrlen;


  if ((b & 0x3F)==0x3F)	// No Command
    return;	    // do nothing

  if ((b & siliCAT_MASK_1STDATA)==siliCAT_1stDATABYTE) // test, if 1st databyte
    rxstate=1;

  else if ((rxstate>0) &&((b & siliCAT_MASK_1STDATA)==siliCAT_nthDATABYTE))  // test, if following databyte
    rxstate++;

  if (rxstate>0)
  {
    if ((b & siliCAT_MASK_DATALM)==siliCAT_BYTEMSB)  // MSB of level-3 byte
        db=b<<6;
    else if (b & siliCAT_BYTELSB) // LSB of level-3 byte
    {
      db|=b& 0x3F;
      switch (rxstate)
      {
        case 1:  // MSB of siliCAT ID
          id=db;
          rxstrbuf[0]=0;
          break;
        case 2:  // LSB of siliCAT ID
          id|=((unsigned int)(db & 0x0F))<<8;
          siliCATTyp=(db>>4)&0x0F;               // siliCAT data type
          if (siliCATTyp==siliCAT_TYPE_BUTTON)   // type: button
          {
            siliCAT_RxValue(id,0,rxstrbuf);
            rxstate=0;
          }
          break;
        case 3:
          siliCATValue=db;
          if (siliCATTyp==siliCAT_TYPE_BYTE)   // type: byte
          {
            siliCAT_RxValue(id,(signed long)(signed char)siliCATValue,rxstrbuf);
            rxstate=0;
          }
          if (siliCATTyp==siliCAT_TYPE_STRING)    // type: string, decode length of string
          {
            rxstrlen=(unsigned char)siliCATValue;
            rxstrbufptr=0;
            rxstate=6;
          }
          break;
        case 4:
          siliCATValue|=(unsigned int)db<<8;
          if (siliCATTyp==siliCAT_TYPE_INT)    // type: integer
          {
            siliCAT_RxValue(id,(signed long)(signed int)siliCATValue,rxstrbuf);
            rxstate=0;
          }
          break;
        case 5:
          siliCATValue|=(unsigned long)db<<16; // 3. byte of long
          break;
        case 6:
          siliCATValue|=(unsigned long)db<<24;
          if (siliCATTyp==siliCAT_TYPE_LONG)   // type long
          {
            siliCAT_RxValue(id,siliCATValue,rxstrbuf);
            rxstate=0;
          }
          break;
        case 7:  // receive string 
          rxstrbuf[rxstrbufptr]=db;
          rxstrbufptr++;
          rxstrbuf[rxstrbufptr]=0;
          rxstate=6;
          if (rxstrbufptr>=rxstrlen)
          {
            siliCAT_RxValue(id,rxstrlen,rxstrbuf);
            rxstate=0;
          }
          break;
      }
    }
  }
}
